<?php
/*
Plugin Name: Wordpess Self-hosted Poster
Plugin URI:
Description: Allows writing posts on your blog connecting remote server with authentication-key and user-key.
Version: 2.0.0
Author: 
Author URI:
License: GPLv2 or later
License URI: http://www.gnu.org/licenses/gpl-2.0.html
*/

defined( 'ABSPATH' ) || exit;

if ( ! function_exists( 'xit_wsh_rewrite_rules_init' ) ) {

    /**
     * Manages rewrite rules
     *
     * @return void
     * @since 1.0.0
     */
    function xit_wsh_rewrite_rules_init() {
        // Generates rewrite rules
        xit_sh_auth_register_rewrites();

        global $wp;

        // Adds some query vars
        $wp->add_query_var( 'xit_wsh_api_v1_endpoint' );
    }

    add_action( 'init', 'xit_wsh_rewrite_rules_init' );
}

if ( ! function_exists( 'xit_sh_auth_register_rewrites' ) ) {

    /**
     * Adds rewrite rule
     *
     * @return void
     * @since 1.0.0
     */
    function xit_sh_auth_register_rewrites() {
        add_rewrite_rule( '^xit/wsh/api/v1/(.+)/?$', 'index.php?xit_wsh_api_v1_endpoint=$matches[1]', 'top' );
    }
}

if ( ! function_exists( 'xit_sh_auth_and_post' ) ) {

    /**
     * Authorizes and inserts post
     *
     * @return void
     * @since 1.0.0
     */
    function xit_sh_auth_and_post()
    {
        $xit_wsh_api_v1_endpoint = filter_var( get_query_var( 'xit_wsh_api_v1_endpoint' ), FILTER_SANITIZE_STRING );
        if ('post' == $xit_wsh_api_v1_endpoint) {
            if ($_POST) {
                // Checks if user and authentication keys exist
                $post_keys = array_keys($_POST);

                if (!in_array('user_key', $post_keys)
                    || !in_array('authentication_key', $post_keys)
                ) {
                    echo json_encode([
                        'status' => 401,
                        'message' => __('Unauthorized', 'xit_wp_sh_auth'),
                    ]);
                    exit;
                }

                $user_key = isset( $_POST['user_key'] )
                    ? filter_var( $_POST['user_key'], FILTER_SANITIZE_STRING )
                    : null;

                $auth_key = isset( $_POST['authentication_key'] )
                    ? filter_var( $_POST['authentication_key'], FILTER_SANITIZE_STRING )
                    : null;

                $meta_key = "xit_wp_sh_auth_key_{$user_key}";

                global $wpdb;
                $value = $wpdb->get_var( $wpdb->prepare( "SELECT meta_value FROM $wpdb->usermeta WHERE meta_key = %s AND meta_value = %s LIMIT 1", $meta_key, $auth_key ) );

                if (null === $value) {
                    echo json_encode( [
                        'status' => 401,
                        'message' => __('Unauthorized', 'xit_wp_sh_auth'),
                    ] );
                    exit;
                }
                
                // Handles request for categories
                if ( isset( $_POST['category'] ) && true == $_POST['category'] ) {
                    $categories = get_categories( array(
                        'taxonomy' => 'category',
                        'orderby' => 'name',
                        'order' => 'ASC',
                        'hide_empty' => true,
                        'include' => 'all',
                        'parent' => false,
                        'fields' => 'all',
                    ) );
                    
                    $category_list = array();
                    if ( count( $categories ) ) {
                        foreach ( $categories as $category ) {
                            $category_list[$category->term_id] = $category->name;
                        }
                    }
                    
                    if ( count( $category_list ) ) {
                        echo json_encode( $category_list );
                        exit;
                    }
                }

                // Tries to insert post into database
                $result = xit_wp_sh_wp_insert_post();

                if ( ! is_array( $result ) ) {
                    echo json_encode( [
                        'status' => 400,
                        'message' => __('Bad request', 'xit_wp_sh_auth'),
                    ] );
                    exit;
                }

                echo json_encode($result);
                exit;
            } else {
                echo json_encode( [
                    'status' => 400,
                    'message' => __('Bad request', 'xit_wp_sh_auth'),
                ] );
                exit;
            }
        }
    }

    add_action( 'template_redirect', 'xit_sh_auth_and_post' );
}

if ( ! function_exists( 'xit_wp_sh_wp_insert_post' ) ) {

    /**
     * Inserts post into database
     *
     * @return array
     * @since 1.0.0
     */
    function xit_wp_sh_wp_insert_post()
    {

        if ( empty( $_POST ) ) {
            return null;
        }

        // Prepares some vars
        $type = isset( $_POST['type'] ) ? $_POST['type'] : null;
        $user_key = isset( $_POST['user_key'] ) ? $_POST['user_key'] : null;
        $post_author = is_string( $user_key ) ? base64_decode( $user_key ) : null;
        $post_author = is_string( $post_author ) ? substr( $post_author, strlen('xit_sh_user_key_') ) : '0';
        $media_url = isset( $_POST['media_url'] ) ? $_POST['media_url'] : '';

        // Decides post type and its content
        $file_url = '';
        $post_mime_type = '';

        if ( 'text' == $type ) {
            $post_content = isset( $_POST['post_content'] ) ? strip_tags( $_POST['post_content'] ) : '';
        } elseif ( 'html' == $type ) {
            $post_content = isset( $_POST['post_content'] ) ? wp_kses_post( html_entity_decode( $_POST['post_content'] ) ) : '';
        } elseif ( 'image' == $type ) {
            $post_content = isset( $_POST['post_content'] ) ? strip_tags( $_POST['post_content'] ) : '';

            // Uploads attachment
            if ( ! empty( $media_url ) ) {
                $status = xit_wp_sh_upload_attachment( $media_url );
                if ( is_array( $status ) ) {
                    $file_url = isset( $status['file_url'] ) ? $status['file_url'] : '';
                    $post_mime_type = isset( $status['post_mime_type'] ) ? $status['post_mime_type'] : '';
                }
            }

        } elseif ( 'video' == $type ) {
            $post_content = isset( $_POST['post_content'] ) ? strip_tags( $_POST['post_content'] ) : '';

            // Uploads attachment
            if ( ! empty( $media_url ) ) {
                $status = xit_wp_sh_upload_attachment( $media_url );
                if ( is_array( $status ) ) {
                    $file_url = isset( $status['file_url'] ) ? $status['file_url'] : '';
                    $post_mime_type = isset( $status['post_mime_type'] ) ? $status['post_mime_type'] : '';
                }
            }
        } else {
            $post_content = isset( $_POST['post_content'] ) ? strip_tags( $_POST['post_content'] ) : '';
        }

        // Prepares post vars
        $post_title = isset( $_POST['post_title'] ) ? sanitize_text_field( $_POST['post_title'] ) : '';
        $post_status = isset( $_POST['post_status'] ) ? sanitize_key( $_POST['post_status']) : 'publish';
        $comment_status = isset( $_POST['comment_status'] ) ? sanitize_key( $_POST['comment_status'] ) : 'open';
        $post_name = isset( $_POST['post_title'] ) ? sanitize_title( $_POST['post_title'] ) : '';
        $post_category = isset( $_POST['post_category'] ) ? $_POST['post_category'] : [];
        $tags_input = isset( $_POST['tags_input'] ) ? $_POST['tags_input'] : [];

        // prepares post data
        $post_data = [
            'post_author' => (int) $post_author,
            'post_content' => $post_content,
            'post_title' => $post_title,
            'post_status' => $post_status,
            'post_type' => 'post',
            'comment_status' => $comment_status,
            'post_name' => $post_name,
            'post_category' => $post_category,
            'tags_input' => $tags_input,
        ];

        // Tries to insert post into database
        $post_id = wp_insert_post( $post_data, true );
        
        // Halts when the post has not been inserted
        if ( ! $post_id > 0 ) {
            return null;
        }

        // Updates attachment post type
        if ( $post_id && $file_url && $post_mime_type ) {
            $data = [
                'type' => $type,
                'post_id' => $post_id,
                'post_title' => $post_title,
                'post_author' => (int)$post_author,
                'post_content' => $post_content,
                'file_url' => $file_url,
                'post_mime_type' => $post_mime_type,
            ];

            xit_wp_sh_insert_attachement( $data );
        }

        $result = [];
        if ( $post_id > 0 ) {
            global $wpdb;
            $result = $wpdb->get_results(
                $wpdb->prepare( "SELECT post_title, post_name, guid FROM $wpdb->posts WHERE ID = %d LIMIT 1", $post_id ),
                ARRAY_A
            );

            if ( is_array( $result ) && count( $result ) > 0 ) {
                $result = $result[0];
            }
        }

        return [
            'post_id' => $post_id,
            'post_title' => isset( $result['post_title'] ) ? $result['post_title'] : '',
            'post_name' => isset( $result['post_name'] ) ? $result['post_name'] : '',
            'url' => isset( $result['guid'] ) ? $result['guid'] : '',
        ];
    }
}

if ( ! function_exists( 'xit_wp_sh_upload_attachment' ) ) {

    /**
     * Uploads media
     *
     * @param string $url
     * @return array|null
     */
    function xit_wp_sh_upload_attachment( $url )
    {
        $basename = basename( parse_url( $url, PHP_URL_PATH ) );
        $sanitized_basename = filter_var( $basename, FILTER_SANITIZE_STRING );

        $wp_upload_path = wp_upload_dir();
        $new_file_abspath = $wp_upload_path['path'] . DIRECTORY_SEPARATOR . $sanitized_basename;

        // Moves the file to destination
        $uploaded = false;
        $old_file = @fopen( $url, 'rb' );
        if ( $old_file ) {
            $new_file = @fopen( $new_file_abspath, 'wb' );
            if ( $new_file ) {
                while ( ! feof( $old_file ) ) {
                    fwrite( $new_file, fread( $old_file, 1024 * 10 ), 1024 * 10 );
                }

                $uploaded = true;
                fclose( $new_file );
            }
            fclose( $old_file );
        }

        if ( true === $uploaded ) {
            $file_url = $wp_upload_path['url'] . DIRECTORY_SEPARATOR . $sanitized_basename;
            $post_mime_type = mime_content_type( $new_file_abspath );

            return [
                'file_url' => $file_url,
                'post_mime_type' => $post_mime_type
            ];
        }

        return null;
    }
}

if ( ! function_exists( 'xit_wp_sh_insert_attachement' ) ) {

    /**
     * Attaches media to post
     *
     * @param array $data
     * @return void
     * @since 1.0.0
     */
    function xit_wp_sh_insert_attachement( $data )
    {
        // Prepare an array of post data for the attachment.
        $attachment = array(
            'guid' => $data['file_url'],
            'post_mime_type' => $data['post_mime_type'],
            'post_author' => $data['post_author'],
            'post_title' => preg_replace('/\.[^.]+$/', '', basename($data['file_url'])),
            'post_content' => $data['post_content'],
            'post_status' => 'inherit'
        );

        // Insert the attachment.
        $attach_id = wp_insert_attachment( $attachment, $data['file_url'], $data['post_id'] );

        // Make sure that this file is included, as wp_generate_attachment_metadata() depends on it.
        require_once( ABSPATH . 'wp-admin/includes/image.php' );
        require_once( ABSPATH . 'wp-admin/includes/file.php' );
        require_once( ABSPATH . 'wp-admin/includes/media.php' );

        // Generate the metadata for the attachment, and update the database record.
        $attach_data = wp_generate_attachment_metadata( $attach_id, $data['file_url'] );
        wp_update_attachment_metadata( $attach_id, $attach_data );

        if ('image' == $data['type']) {
            set_post_thumbnail( $data['post_id'], $attach_id );
        }
    }
}

if ( ! function_exists( 'xit_wsh_api_v1_get_posts' ) ) {
    
    /**
     * Gets published posts
     *
     * @return void
     * @since 2.0.0
     */
    function xit_wsh_api_v1_get_posts() 
    {
        $xit_wsh_api_v1_endpoint = filter_var( get_query_var( 'xit_wsh_api_v1_endpoint' ), FILTER_SANITIZE_STRING );
        if ('posts' == $xit_wsh_api_v1_endpoint) {
            
            $args = [
                'numberposts'       => 10,
                'category'          => 0,
                'orderby'           => 'date',
                'order'             => 'DESC',
            ];
            
            $posts = get_posts( $args );
            
            wp_send_json( $posts );
        }
    }
    
    add_action( 'template_redirect', 'xit_wsh_api_v1_get_posts' );
}

if ( ! function_exists( 'xit_wsh_api_v1_get_categories' ) ) {
    
    /**
     * Gets categories
     *
     * @return void
     * @since 2.0.0
     */
    function xit_wsh_api_v1_get_categories() 
    {
        $xit_wsh_api_v1_endpoint = filter_var( get_query_var( 'xit_wsh_api_v1_endpoint' ), FILTER_SANITIZE_STRING );
        if ('categories' == $xit_wsh_api_v1_endpoint) {
            
            $args = [
                'hide_empty'        => false,
                'orderby'           => 'name',
                'order'             => 'ASC',
                'parent'            => false,
            ];
            
            $categories = get_categories( $args );
            
            wp_send_json( $categories );
        }
    }
    
    add_action( 'template_redirect', 'xit_wsh_api_v1_get_categories' );
}

if ( ! function_exists( 'xit_wp_sh_authentication_field' ) ) {

    /**
     * Displays fields onto user profile
     *
     * @param WP_User $user
     * @return bool|void
     * @since 1.0.0
     */
    function xit_wp_sh_authentication_field($user)
    {

        if (!current_user_can('administrator', $user->ID)) {
            return false;
        }

        $hashed_user_id = base64_encode( 'xit_sh_user_key_' . $user->ID );

        $auth_meta_key = 'xit_wp_sh_auth_key_' . $hashed_user_id;
        $xit_wp_sh_auth_key = get_user_meta($user->ID, $auth_meta_key, true);

        ?>

        <h3><?php esc_html_e('Wordpress Self-hosted Authentication', 'xit_wp_sh_auth'); ?></h3>
        <p class="description"><?php echo esc_html_e('Expects a post request with "user_key", "authentication_key" and post data.', 'xit_wp_sh_auth'); ?></p>

        <table class="form-table">
            <tr>
                <th>
                    <label for="xit_wp_sh_auth_key_<?php echo $hashed_user_id; ?>"><?php esc_html_e('Authentication Key', 'xit_wp_sh_auth'); ?></label>
                </th>
                <td>
                    <input type="text"
                           id="xit_wp_sh_auth_key_<?php echo $hashed_user_id; ?>"
                           name="xit_wp_sh_auth_key_<?php echo $hashed_user_id; ?>"
                           value="<?php echo esc_attr($xit_wp_sh_auth_key); ?>"
                           class="regular-text"
                    />
                    <p class="description"><?php echo esc_html_e('Put in the Authentication Key which is available from where you download this plugin.', 'xit_wp_sh_auth'); ?></p>
                </td>
            </tr>
            <tr>
                <th><label for="xit_wp_sh_user_key"><?php esc_html_e('User Key', 'xit_wp_sh_auth'); ?></label></th>
                <td>
                    <input type="text"
                           id="xit_wp_sh_user_key"
                           name="xit_wp_sh_user_key"
                           value="<?php echo esc_attr($hashed_user_id); ?>"
                           class="regular-text"
                           disabled
                    />
                    <p class="description"><?php echo esc_html_e('Copy this code and put in the specified box from where you get the Authentication Key.', 'xit_wp_sh_auth'); ?></p>
                </td>
            </tr>
        </table>
        <?php
    }

    add_action( 'show_user_profile', 'xit_wp_sh_authentication_field' );
    add_action( 'edit_user_profile', 'xit_wp_sh_authentication_field' );
}

if ( ! function_exists( 'xit_sh_auth_update_profile_field' ) ) {

    /**
     * Updates user provided data
     *
     * @param $user_id
     * @return bool|void
     * @since 1.0.0
     */
    function xit_sh_auth_update_profile_field($user_id)
    {

        if (!current_user_can('administrator', $user_id)) {
            return false;
        }

        $hashed_user_id = base64_encode('xit_sh_user_key_' . $user_id);
        $auth_key = "xit_wp_sh_auth_key_{$hashed_user_id}";
        $webhook_url = "xit_wp_sh_webhook_url_{$hashed_user_id}";

        if ( ! empty($_POST[$auth_key]) && sanitize_text_field( $_POST[$auth_key] ) ) {
            update_user_meta( $user_id, $auth_key, $_POST[$auth_key] );
        }
    }

    add_action('personal_options_update', 'xit_sh_auth_update_profile_field');
    add_action('edit_user_profile_update', 'xit_sh_auth_update_profile_field');
}

if ( ! function_exists( 'xit_sh_auth_activation' ) ) {

    /**
     * Runs on plugin activation
     *
     * @param bool $network_wide
     * @return void
     * @since 1.0.0
     */
    function xit_sh_auth_activation( $network_wide )
    {
        if (function_exists( 'is_multisite') && is_multisite() && $network_wide ) {

            $mu_blogs = wp_get_sites();

            foreach ( $mu_blogs as $mu_blog ) {
                switch_to_blog( $mu_blog['blog_id'] );
                xit_sh_auth_register_rewrites();
                flush_rewrite_rules();
            }

            restore_current_blog();
        } else {
            xit_sh_auth_register_rewrites();
            flush_rewrite_rules();
        }
    }

    register_activation_hook( __FILE__, 'xit_sh_auth_activation' );
}

if ( ! function_exists( 'xit_sh_auth_deactivation' ) ) {

    /**
     * Runs on plugin deactivation
     *
     * @param bool $network_wide
     * @return void
     * @since 1.0.0
     */
    function xit_sh_auth_deactivation( $network_wide )
    {
        if (function_exists( 'is_multisite') && is_multisite() && $network_wide ) {

            $mu_blogs = wp_get_sites();

            foreach ( $mu_blogs as $mu_blog ) {
                switch_to_blog($mu_blog['blog_id']);
                flush_rewrite_rules();
            }

            restore_current_blog();
        } else {
            flush_rewrite_rules();
        }
    }

    register_deactivation_hook( __FILE__, 'xit_sh_auth_deactivation' );
}
